module ContortHelperBurstZone

using ..Ahorn, Maple

# Mapdef

@mapdef Entity "ContortHelper/BurstZone" BurstZone(
	x::Integer,
	y::Integer,
	width::Integer=Maple.defaultBlockWidth,
	height::Integer=Maple.defaultBlockHeight,

	textures::String="util/displacementcircle",

	burstDurationMinimum::Number=1.5,
	burstDurationMaximum::Number=1.5,

	coroutineDurationMinimum::Number=0.01,
	coroutineDurationMaximum::Number=0.01,

	radiusFromMinimum::Number=0.0,
	radiusFromMaximum::Number=0.0,

	radiusToMinimum::Number=25.0,
	radiusToMaximum::Number=25.0,

	alphaMinimum::Number=1.0,
	alphaMaximum::Number=1.0,

	alphaEaser::String="CubeInOut",
	radiusEaser::String="CubeInOut",

	amountPerUpdate::Integer=1,

	needed::Bool=false
)

# Placements

const placements = Ahorn.PlacementDict(
	"Burst Zone ($(name)) (Contort Helper)" => Ahorn.EntityPlacement(
		BurstZone,
		"rectangle",
		Dict{String,Any}(
			"burstDurationMinimum" => burstDuration,
			"burstDurationMaximum" => burstDuration,
			"coroutineDurationMinimum" => coroutineDuration,
			"coroutineDurationMaximum" => coroutineDuration,
			"radiusToMinimum" => radiusTo,
			"radiusToMaximum" => radiusTo
		)
	) for (name, burstDuration, coroutineDuration, radiusTo) in Ahorn.ContortHelper.burstzonevariants
)

# BurstZone

Ahorn.minimumSize(entity::BurstZone) = 8, 8
Ahorn.resizable(entity::BurstZone) = true, true

const color = (0.4, 0.6, 1.0, 0.4)
const outline = (0.5, 1.0, 1.0, 0.6)
const text = "Burst Zone"

function Ahorn.selection(entity::BurstZone, room::Maple.Room, node::Int=0)
	x, y = Int(entity.x), Int(entity.y)
	width, height = Int(entity.width), Int(entity.height)
	nodes = get(entity.data, "nodes", Tuple{Int, Int}[])

	if isempty(nodes)
		return Ahorn.Rectangle(x, y, width, height)

	else
		res = Ahorn.Rectangle[Ahorn.Rectangle(x, y, width, height)]

		for node in nodes
			nx, ny = Int.(node)

			push!(res, Ahorn.Rectangle(nx, ny, 8, 8))
		end

		return res
	end
end

function Ahorn.render(ctx::Ahorn.Cairo.CairoContext, entity::BurstZone, room::Maple.Room)
	if ctx.ptr != C_NULL
		Ahorn.Cairo.save(ctx)

		x, y = Ahorn.position(entity)
		w, h = Int(entity.width), Int(entity.height)
		
		x /= typemax(Int)
		y /= typemax(Int)

		Ahorn.rectangle(ctx, x, y, w, h)
		Ahorn.clip(ctx)

		Ahorn.drawRectangle(ctx, x, y, w, h, color, outline)
		Ahorn.drawCenteredText(ctx, text, x, y, w, h)

		Ahorn.restore(ctx)
	end
end

function Ahorn.renderSelectedAbs(ctx::Ahorn.Cairo.CairoContext, entity::BurstZone)
	x, y = Int(entity.data["x"]), Int(entity.data["y"])
	width, height = Int(entity.data["width"]), Int(entity.data["height"])
	nodes = get(entity.data, "nodes", Tuple{Int, Int}[])
	offsetCenterX, offsetCenterY = floor(Int, width / 2), floor(Int, height / 2)

	for node in nodes
		nx, ny = Int.(node)

		Ahorn.drawArrow(ctx, x + offsetCenterX, y + offsetCenterY, nx + 4, ny + 4, outline, headLength=6)
		Ahorn.drawRectangle(ctx, nx, ny, 8, 8, color, outline)
	end
end

Ahorn.editingOptions(entity::BurstZone) = Dict{String, Any}(
	"alphaEaser" => Ahorn.ContortHelper.easers,
	"radiusEaser" => Ahorn.ContortHelper.easers
)

end