module ContortHelperLightSourceZone

using ..Ahorn, Maple

# Mapdef

@mapdef Entity "ContortHelper/LightSourceZone" LightSourceZone(
	x::Integer,
	y::Integer,
	width::Integer=Maple.defaultBlockWidth,
	height::Integer=Maple.defaultBlockHeight,

	colors::String="ffffff,ffffff,ffffff",

	alphaMinimum::Number=0.8,
	alphaMaximum::Number=1.0,

	radiusMinimum::Number=46.0,
	radiusMaximum::Number=48.0,

	startFadeMinimum::Number=22.0,
	startFadeMaximum::Number=24.0,

	endFadeMinimum::Number=46.0,
	endFadeMaximum::Number=48.0,

	amount::Integer=5
)

# Placements

const placements = Ahorn.PlacementDict(
	"Light Zone ($(name)) (Contort Helper)" => Ahorn.EntityPlacement(
		LightSourceZone,
		"rectangle",
		Dict{String,Any}(
			"startFadeMinimum" => startFade,
			"startFadeMaximum" => startFade,
			"endFadeMinimum" => endFade,
			"endFadeMaximum" => endFade,
			"radiusMinimum" => radius,
			"radiusMaximum" => radius
		)
	) for (name, startFade, endFade, radius) in Ahorn.ContortHelper.lightvariants
)

# LightSourceZone

Ahorn.minimumSize(entity::LightSourceZone) = 8, 8
Ahorn.resizable(entity::LightSourceZone) = true, true

const color = (0.9, 0.5, 0.8, 0.4)
const outline = (1.0, 0.8, 1.0, 0.6)
const text = "Light Zone"

function Ahorn.selection(entity::LightSourceZone, room::Maple.Room, node::Int=0)
	x, y = Int(entity.x), Int(entity.y)
	width, height = Int(entity.width), Int(entity.height)
	nodes = get(entity.data, "nodes", Tuple{Int, Int}[])

	if isempty(nodes)
		return Ahorn.Rectangle(x, y, width, height)

	else
		res = Ahorn.Rectangle[Ahorn.Rectangle(x, y, width, height)]

		for node in nodes
			nx, ny = Int.(node)

			push!(res, Ahorn.Rectangle(nx, ny, 8, 8))
		end

		return res
	end
end

function Ahorn.render(ctx::Ahorn.Cairo.CairoContext, entity::LightSourceZone, room::Maple.Room)
	if ctx.ptr != C_NULL
		Ahorn.Cairo.save(ctx)

		x, y = Ahorn.position(entity)
		w, h = Int(entity.width), Int(entity.height)
		
		x /= typemax(Int)
		y /= typemax(Int)

		Ahorn.rectangle(ctx, x, y, w, h)
		Ahorn.clip(ctx)

		Ahorn.drawRectangle(ctx, x, y, w, h, color, outline)
		Ahorn.drawCenteredText(ctx, text, x, y, w, h)

		Ahorn.restore(ctx)
	end
end

function Ahorn.renderSelectedAbs(ctx::Ahorn.Cairo.CairoContext, entity::LightSourceZone)
	x, y = Int(entity.data["x"]), Int(entity.data["y"])
	width, height = Int(entity.data["width"]), Int(entity.data["height"])
	nodes = get(entity.data, "nodes", Tuple{Int, Int}[])
	offsetCenterX, offsetCenterY = floor(Int, width / 2), floor(Int, height / 2)

	for node in nodes
		nx, ny = Int.(node)

		Ahorn.drawArrow(ctx, x + offsetCenterX, y + offsetCenterY, nx + 4, ny + 4, outline, headLength=6)
		Ahorn.drawRectangle(ctx, nx, ny, 8, 8, color, outline)
	end
end

Ahorn.editingOptions(entity::LightSourceZone) = Dict{String,Any}(
	"colors" => sort(collect(keys(Ahorn.XNAColors.colors)))
)

end