// #include "Common.fxh"
// Platform-Specific Definitions
// for XNA/FNA/MonoGame:

#define DECLARE_TEXTURE(Name, index) \
    texture Name: register(t##index); \
    sampler Name##Sampler: register(s##index)

#define SAMPLE_TEXTURE(Name, texCoord) tex2D(Name##Sampler, texCoord)

#define PS_3_SHADER_COMPILER ps_3_0
#define PS_2_SHADER_COMPILER ps_2_0
#define VS_SHADER_COMPILER vs_3_0
#define VS_2_SHADER_COMPILER vs_2_0

#define SV_TARGET0 COLOR0
#define SV_TARGET1 COLOR1
#define SV_TARGET2 COLOR2

//-----------------------------------------------------------------------------
// Globals.
//-----------------------------------------------------------------------------

DECLARE_TEXTURE(text, 0);
DECLARE_TEXTURE(map, 1);
uniform float anxiety = 0.0;
uniform float2 anxietyOrigin = float2(0.5, 0.5);
uniform float gamerate = 1.0;
uniform float waterSine = 0.0;
uniform float waterCameraY = 0.0;
uniform float waterAlpha = 1.0;
uniform float2 bufferSize = float2(320.0, 180.0);

//-----------------------------------------------------------------------------
// Pixel Shaders.
//-----------------------------------------------------------------------------

float2 GetDisplacement(float2 texcoord)
{
    float2 pixelatedTexcoord = float2(floor(texcoord.x * bufferSize.x) / bufferSize.x + 0.5 / bufferSize.x, floor(texcoord.y * bufferSize.y) / bufferSize.y + 0.5 / bufferSize.y);

    // normal displacement
    float4 displacementPixel = SAMPLE_TEXTURE(map, pixelatedTexcoord);
    float shift = waterAlpha * sin((pixelatedTexcoord.y * bufferSize.y + waterCameraY) * 0.3 - waterSine * displacementPixel.b) * 0.004;

    if (
        abs((displacementPixel.r * 2.0 - 1.0) * 0.044) < 0.49 / bufferSize.x
        && abs((displacementPixel.g * 2.0 - 1.0) * 0.078) < 0.49 / bufferSize.y
        && abs(shift * ceil(displacementPixel.b)) < 0.1 / bufferSize.x
    ) {
        displacementPixel = float4(0.5, 0.5, 0.0, 0.0);
        pixelatedTexcoord = texcoord;
    }

    float2 position = pixelatedTexcoord;
    position.x += (displacementPixel.r * 2.0 - 1.0) * 0.044;
    position.y += (displacementPixel.g * 2.0 - 1.0) * 0.078;

    // water shifting stuff
    // amount of BLUE describes how FAST it should wave (range 0.0 -> 1.0)
    position.x += shift * ceil(displacementPixel.b);

    return position;
}

float4 GetAnxietyColor(float2 texcoord)
{
    // get anxiety amount
    float len = length(texcoord - anxietyOrigin) * 2.0;
    float anx = 0.02 * len * anxiety;

    // offset R & B samples by anxiety amount
    float4 r = SAMPLE_TEXTURE(text, texcoord + float2(anx, 0.0));
    float4 g = SAMPLE_TEXTURE(text, texcoord + float2(0.0, 0.0));
    float4 b = SAMPLE_TEXTURE(text, texcoord + float2(-anx, 0.0));
    
    return float4(r.x, g.y, b.z, b.w);
}

float4 GetGrayscaleColor(float4 color)
{
    // gamerate -> black & white
    float gray = float(color.r * 0.3 + color.g * 0.59 + color.b * 0.11);
    return lerp(color, float4(gray, gray, gray, color.w), 1 - gamerate);
}

// displacement and anxiety
float4 PS_Distort(float4 inPosition : SV_Position, float4 inColor : COLOR0, float2 uv : TEXCOORD0) : SV_TARGET0
{
    return GetGrayscaleColor(GetAnxietyColor(GetDisplacement(uv)));
}

// only do displacement
float4 PS_Displace(float4 inPosition : SV_Position, float4 inColor : COLOR0, float2 uv : TEXCOORD0) : SV_TARGET0
{
    return SAMPLE_TEXTURE(text, GetDisplacement(uv));
}

//-----------------------------------------------------------------------------
// Techniques.
//-----------------------------------------------------------------------------

technique Distort
{
    pass
    {
        PixelShader = compile PS_3_SHADER_COMPILER PS_Distort();
    }
}

technique Displace
{
    pass
    {
        PixelShader = compile PS_3_SHADER_COMPILER PS_Displace();
    }
}