module ChronoHelperCustomTimeSwitchGates
using ..Ahorn, Maple

function gateFinalizer(entity)
    x, y = Ahorn.position(entity)

    width = Int(get(entity.data, "width", 8))
    height = Int(get(entity.data, "height", 8))

    entity.data["nodes"] = [(x + width, y)]
end

@mapdef Entity "ChronoHelper/CustomTimeSwitchGates" CustomTimeSwitchGates(
			x::Integer, y::Integer,
			width::Integer=8, height::Integer=8,
			sprite::String="block",
			persistent::Bool=false,
			moveTime::Number=2.0,
            shakeTime::Number=0.5,
            preShakeTime::Number=0.1,
            postShakeTime::Number=0.1,
            moveEased::Bool=true,
			nodes::Array{Tuple{Integer, Integer}, 1}=Tuple{Integer, Integer}[])
			

const textures = ["block", "mirror", "temple", "stars"]

const placements = Ahorn.PlacementDict(
    "Custom Time Switch Gates ($(uppercasefirst(texture))) (Chrono Helper)" => Ahorn.EntityPlacement(
        CustomTimeSwitchGates,
		"rectangle",
		Dict{String, Any}(
			"sprite" => texture
        ),
        gateFinalizer
    ) for texture in textures
)

Ahorn.editingOptions(entity::CustomTimeSwitchGates) = Dict{String, Any}(
    "sprite" => textures,
)

Ahorn.nodeLimits(entity::CustomTimeSwitchGates) = 1, 1
Ahorn.minimumSize(entity::CustomTimeSwitchGates) = 16, 16
Ahorn.resizable(entity::CustomTimeSwitchGates) = true, true

function Ahorn.selection(entity::CustomTimeSwitchGates)
    x, y = Ahorn.position(entity)
    stopX, stopY = Int.(entity.data["nodes"][1])

    width = Int(get(entity.data, "width", 8))
    height = Int(get(entity.data, "height", 8))

    return [Ahorn.Rectangle(x, y, width, height), Ahorn.Rectangle(stopX, stopY, width, height)]
end

iconResource = "objects/switchgate/icon00"

function renderGateSwitch(ctx::Ahorn.Cairo.CairoContext, x::Number, y::Number, width::Number, height::Number, sprite::String)
    iconSprite = Ahorn.getSprite(iconResource, "Gameplay")
    
    tilesWidth = div(width, 8)
    tilesHeight = div(height, 8)

    frame = "objects/switchgate/$sprite"

    for i in 2:tilesWidth - 1
        Ahorn.drawImage(ctx, frame, x + (i - 1) * 8, y, 8, 0, 8, 8)
        Ahorn.drawImage(ctx, frame, x + (i - 1) * 8, y + height - 8, 8, 16, 8, 8)
    end

    for i in 2:tilesHeight - 1
        Ahorn.drawImage(ctx, frame, x, y + (i - 1) * 8, 0, 8, 8, 8)
        Ahorn.drawImage(ctx, frame, x + width - 8, y + (i - 1) * 8, 16, 8, 8, 8)
    end

    for i in 2:tilesWidth - 1, j in 2:tilesHeight - 1
        Ahorn.drawImage(ctx, frame, x + (i - 1) * 8, y + (j - 1) * 8, 8, 8, 8, 8)
    end

    Ahorn.drawImage(ctx, frame, x, y, 0, 0, 8, 8)
    Ahorn.drawImage(ctx, frame, x + width - 8, y, 16, 0, 8, 8)
    Ahorn.drawImage(ctx, frame, x, y + height - 8, 0, 16, 8, 8)
    Ahorn.drawImage(ctx, frame, x + width - 8, y + height - 8, 16, 16, 8, 8)

    Ahorn.drawImage(ctx, iconSprite, x + div(width - iconSprite.width, 2), y + div(height - iconSprite.height, 2))
end

function Ahorn.renderSelectedAbs(ctx::Ahorn.Cairo.CairoContext, entity::CustomTimeSwitchGates, room::Maple.Room)
	
    sprite = get(entity.data, "sprite", "block")
    startX, startY = Int(entity.data["x"]), Int(entity.data["y"])
    stopX, stopY = Int.(entity.data["nodes"][1])

    width = Int(get(entity.data, "width", 32))
    height = Int(get(entity.data, "height", 32))

    renderGateSwitch(ctx, stopX, stopY, width, height, sprite)
    Ahorn.drawArrow(ctx, startX + width / 2, startY + height / 2, stopX + width / 2, stopY + height / 2, Ahorn.colors.selection_selected_fc, headLength=6)
end

function Ahorn.renderAbs(ctx::Ahorn.Cairo.CairoContext, entity::CustomTimeSwitchGates, room::Maple.Room)
    sprite = get(entity.data, "sprite", "block")

    x = Int(get(entity.data, "x", 0))
    y = Int(get(entity.data, "y", 0))

    width = Int(get(entity.data, "width", 32))
    height = Int(get(entity.data, "height", 32))

    renderGateSwitch(ctx, x, y, width, height, sprite)
end

end
