#define DECLARE_TEXTURE(Name, index) \
    texture Name: register(t##index); \
    sampler Name##Sampler: register(s##index)

#define SAMPLE_TEXTURE(Name, texCoord) tex2D(Name##Sampler, texCoord)

uniform float Time; // level.TimeActive
uniform float2 CamPos; // level.Camera.Position
uniform float2 Dimensions; // new Vector2(320, 180)

uniform float4x4 TransformMatrix;
uniform float4x4 ViewMatrix;

DECLARE_TEXTURE(text, 0);

float fracDot(float2 a, float2 b){
    return frac(sin(dot(a, b)) * 43758.5453123);
}

float noise(float2 p){
    return fracDot(p, float2(127.1, 311.7));
}

float smoothNoise(float2 p){
    float2 i = floor(p);
    float2 f = frac(p);
    float2 u = f * f * (3.0 - 2.0 * f);

    float n00 = noise(i);
    float n10 = noise(i + float2(1.0, 0.0));
    float n01 = noise(i + float2(0.0, 1.0));
    float n11 = noise(i + float2(1.0, 1.0));

    float x1 = lerp(n00, n10, u.x);
    float x2 = lerp(n01, n11, u.x);
    return lerp(x1, x2, u.y);
}

float fbm(float2 p){
    float total = 0.0;
    float amplitude = 0.5;
    float frequency = 1.0;

    [unroll]
    for (int i = 0; i < 2; i++)
    {
        total += amplitude * smoothNoise(p * frequency);
        frequency *= 2.0;
        amplitude *= 0.5;
    }

    return total;
}

float4 SpritePixelShader(float2 uv : TEXCOORD0) : COLOR0
{
    float2 worldPos = (uv * Dimensions) + CamPos;
    float2 worldPosAlt = (uv * Dimensions) + (CamPos / 1.7);
    float3 color = SAMPLE_TEXTURE(text, uv).rgb;

    float2 samplepos = uv;
    float3 sampled = color;
    float dir = 0;
    float dist = 1;

    for (int i = 0; i < 8; i++){
        for (int i = 0; i < 32; i++){
            samplepos = float2(uv.x + (sin(dir) * 0.005 * dist),uv.y + (cos(dir) * 0.01 * dist));
            sampled = SAMPLE_TEXTURE(text, samplepos).rgb;
            if ((sampled.g / 1.4) + sampled.b - sampled.r > 1.2){
                color.gb += 0.008 / dist;
                color.r += 0.004 / dist;
            }
            if (sampled.r + sampled.b - sampled.g > 1.2){
                color.rb += 0.008 / dist;
                color.g += 0.004 / dist;
            }
            dir += 1;
        }
        dist += 1;
    }

    float mist = fbm(float2(worldPos.x + (Time * 20),worldPos.y) / 30);
    color.rb += (mist / 4) * uv.y;
    color.g += (mist / 12) * uv.y;

    float beams = fbm(float2(worldPos.x + (Time * 15),worldPos.x - (Time * 15)) / 20);
    color.gb += (beams / 7) * (1 - uv.y);
    color.r += (beams / 9) * (1 - uv.y);

    float brightness = (color.r + color.g + color.b) / 3;
    color = lerp(color * brightness,color,0.6);
    color = lerp(color,float3(brightness,brightness,brightness),0.12);

    float particles = fbm(float2(worldPosAlt.x + (Time * 15),worldPosAlt.y) / 100);

    if (particles + mist > 0.8){
        color.b += (particles + mist - 0.8) / 2;
        color.rg += (particles + mist - 0.8) / 3;
    }

    color.rb += uv.y / 10;
    color.gb += (1 - uv.y) / 10;

    return float4(color,1.0);
}

void SpriteVertexShader(inout float4 color    : COLOR0,
                        inout float2 texCoord : TEXCOORD0,
                        inout float4 position : SV_Position)
{
    position = mul(position, ViewMatrix);
    position = mul(position, TransformMatrix);
}

technique Shader
{
    pass pass0
    {
        VertexShader = compile vs_3_0 SpriteVertexShader();
        PixelShader = compile ps_3_0 SpritePixelShader();
    }
}