#define DECLARE_TEXTURE(Name, index) \
    texture Name: register(t##index); \
    sampler Name##Sampler: register(s##index)

#define SAMPLE_TEXTURE(Name, texCoord) tex2D(Name##Sampler, texCoord)

uniform float Time; // level.TimeActive
uniform float2 CamPos; // level.Camera.Position
uniform float2 Dimensions; // new Vector2(320, 180)

uniform float4x4 TransformMatrix;
uniform float4x4 ViewMatrix;

DECLARE_TEXTURE(text, 0);

float fracDot(float2 a, float2 b){
    return frac(sin(dot(a, b)) * 43758.5453123);
}

float noise(float2 p){
    return fracDot(p, float2(127.1, 311.7));
}

float smoothNoise(float2 p){
    float2 i = floor(p);
    float2 f = frac(p);
    float2 u = f * f * (3.0 - 2.0 * f);

    float n00 = noise(i);
    float n10 = noise(i + float2(1.0, 0.0));
    float n01 = noise(i + float2(0.0, 1.0));
    float n11 = noise(i + float2(1.0, 1.0));

    float x1 = lerp(n00, n10, u.x);
    float x2 = lerp(n01, n11, u.x);
    return lerp(x1, x2, u.y);
}

float fbm(float2 p){
    float total = 0.0;
    float amplitude = 0.5;
    float frequency = 1.0;

    [unroll]
    for (int i = 0; i < 6; i++)
    {
        total += amplitude * smoothNoise(p * frequency);
        frequency *= 2.0;
        amplitude *= 0.5;
    }

    return total;
}

float4 SpritePixelShader(float2 uv : TEXCOORD0) : COLOR0
{
    float2 worldPos = (uv * Dimensions) + CamPos;
    float3 color = SAMPLE_TEXTURE(text, uv).rgb;

    float3 sampled = color;
    float2 samplepos = uv;
    float dist = 1;
    float strength = 1;

    color.r *= 1.2;
    color.g *= 1.03;

    float random = fbm(worldPos / 60);
    color.r += random / 12;
    color.g += random / 20;
    color.b += random / 30;

    for (int i = 0; i < 24; i++){
        samplepos = ((uv - 0.5) / dist) + 0.5;
        sampled = SAMPLE_TEXTURE(text, samplepos).rgb;
        if (sampled.r + sampled.g + sampled.b > 1.8){
            color += 0.02 * strength;
            color.r += 0.02 * strength;
        }
        strength /= 1.12;
        dist += 0.02;
    }

    return float4(color,SAMPLE_TEXTURE(text, uv).a); 
}

void SpriteVertexShader(inout float4 color    : COLOR0,
                        inout float2 texCoord : TEXCOORD0,
                        inout float4 position : SV_Position)
{
    position = mul(position, ViewMatrix);
    position = mul(position, TransformMatrix);
}

technique Shader
{
    pass pass0
    {
        VertexShader = compile vs_3_0 SpriteVertexShader();
        PixelShader = compile ps_3_0 SpritePixelShader();
    }
}