﻿using System;
using System.Runtime.CompilerServices;
using Celeste.Mod.Entities;
using FMOD.Studio;
using Iced.Intel;
using Microsoft.Xna.Framework;
using Monocle;

namespace Celeste;

[Tracked(false)]
public class TemporarySwapBlock : Solid
{
    public enum Themes
    {
        Normal,
        Moon
    }

    private class PathRenderer : Entity
    {
        private TemporarySwapBlock block;

        private MTexture pathTexture;

        private MTexture clipTexture = new MTexture();

        private float timer;

        [MethodImpl(MethodImplOptions.NoInlining)]
        public PathRenderer(TemporarySwapBlock block)
            : base(block.Position)
        {
            this.block = block;
            base.Depth = 8999;
            pathTexture = GFX.Game["objects/swapblock/path" + ((block.start.X == block.end.X) ? "V" : "H")];
            timer = Calc.Random.NextFloat();
        }

        [MethodImpl(MethodImplOptions.NoInlining)]
        public override void Update()
        {
            base.Update();
            if (block == null)
            {
                RemoveSelf();
            }
            timer += Engine.DeltaTime * 4f;
        }

        [MethodImpl(MethodImplOptions.NoInlining)]
        public override void Render()
        {

            float num = 0.5f * (0.5f + ((float)Math.Sin(timer) + 1f) * 0.25f);
            block.DrawBlockStyle(new Vector2(block.moveRect.X, block.moveRect.Y), block.moveRect.Width, block.moveRect.Height, block.nineSliceTarget, null, Color.White * num);
        }
    }

    private const float ReturnTime = 0.8f;

    public Vector2 Direction;

    public bool Swapping;

    public Themes Theme;

    private Vector2 start;

    private Vector2 end;

    private float lerp;

    private int target;

    private Rectangle moveRect;

    private float speed;

    private float maxForwardSpeed;

    private float maxBackwardSpeed;

    private float returnTimer;

    private float redAlpha = 1f;

    private MTexture[,] nineSliceGreen;

    private MTexture[,] nineSliceRed;

    private MTexture[,] nineSliceTarget;

    private Sprite middleGreen;

    private Sprite middleRed;

    private PathRenderer path;

    private EventInstance moveSfx;

    private EventInstance returnSfx;

    private DisplacementRenderer.Burst burst;

    private float particlesRemainder;

    [MethodImpl(MethodImplOptions.NoInlining)]

    public TemporarySwapBlock Init(Vector2 position, float width, float height, Vector2 node)
    {
        start = Position;
        end = node;
        maxForwardSpeed = 360f / Vector2.Distance(start, end);
        maxBackwardSpeed = maxForwardSpeed * 0.4f;
        Direction.X = Math.Sign(end.X - start.X);
        Direction.Y = Math.Sign(end.Y - start.Y);
        Add(new DashListener
        {
            OnDash = OnDash
        });
        int num = (int)MathHelper.Min(base.X, node.X);
        int num2 = (int)MathHelper.Min(base.Y, node.Y);
        int num3 = (int)MathHelper.Max(base.X + base.Width, node.X + base.Width);
        int num4 = (int)MathHelper.Max(base.Y + base.Height, node.Y + base.Height);
        moveRect = new Rectangle(num, num2, num3 - num, num4 - num2);
        MTexture mTexture;
        MTexture mTexture2;
        MTexture mTexture3;
        mTexture = GFX.Game["objects/swapblock/moon/block"];
        mTexture2 = GFX.Game["objects/swapblock/moon/blockRed"];
        mTexture3 = GFX.Game["objects/swapblock/moon/target"];

        nineSliceGreen = new MTexture[3, 3];
        nineSliceRed = new MTexture[3, 3];
        nineSliceTarget = new MTexture[3, 3];
        for (int i = 0; i < 3; i++)
        {
            for (int j = 0; j < 3; j++)
            {
                nineSliceGreen[i, j] = mTexture.GetSubtexture(new Rectangle(i * 8, j * 8, 8, 8));
                nineSliceRed[i, j] = mTexture2.GetSubtexture(new Rectangle(i * 8, j * 8, 8, 8));
                nineSliceTarget[i, j] = mTexture3.GetSubtexture(new Rectangle(i * 8, j * 8, 8, 8));
            }
        }
        Add(middleGreen = GFX.SpriteBank.Create("swapBlockLightMoon"));
        Add(middleRed = GFX.SpriteBank.Create("swapBlockLightRedMoon"));

        Add(new LightOcclude(0.2f));
        base.Depth = -9999;
        return this;
    }
    public TemporarySwapBlock(Vector2 position, float width, float height, bool safe) : base(position, width, height,false)
    {
        Position = position;
    }

    [MethodImpl(MethodImplOptions.NoInlining)]
    public override void Awake(Scene scene)
    {
        base.Awake(scene);
        scene.Add(path = new PathRenderer(this));
    }

    [MethodImpl(MethodImplOptions.NoInlining)]
    public override void Removed(Scene scene)
    {
        base.Removed(scene);
        Audio.Stop(moveSfx);
        Audio.Stop(returnSfx);
    }

    [MethodImpl(MethodImplOptions.NoInlining)]
    public override void SceneEnd(Scene scene)
    {
        base.SceneEnd(scene);
        Audio.Stop(moveSfx);
        Audio.Stop(returnSfx);
    }

    [MethodImpl(MethodImplOptions.NoInlining)]
    public void OnDash(Vector2 direction)
    {
        Swapping = lerp < 1f;
        target = 1;
        returnTimer = 0.8f;
        //SceneAs<Level>().Displacement.AddBurst(Center, 0.2f, 0f, 16f);
        if (lerp >= 0.2f)
        {
            speed = maxForwardSpeed;
        }
        else
        {
            speed = MathHelper.Lerp(maxForwardSpeed * 0.333f, maxForwardSpeed, lerp / 0.2f);
        }

        Audio.Stop(returnSfx);
        Audio.Stop(moveSfx);
        if (!Swapping)
        {
            Audio.Play("event:/game/05_mirror_temple/swapblock_move_end", base.Center);
        }
        else
        {
            moveSfx = Audio.Play("event:/game/05_mirror_temple/swapblock_move", base.Center);
        }
    }

    [MethodImpl(MethodImplOptions.NoInlining)]
    public override void Update()
    {
        base.Update();
        if (returnTimer > 0f)
        {
            returnTimer -= Engine.DeltaTime;
            if (returnTimer <= 0f)
            {
                target = 0;
                speed = 0f;
                returnSfx = Audio.Play("event:/game/05_mirror_temple/swapblock_return", base.Center);
            }
        }

        if (burst != null)
        {
            burst.Position = base.Center;
        }

        redAlpha = Calc.Approach(redAlpha, (target != 1) ? 1 : 0, Engine.DeltaTime * 32f);
        if (target == 0 && lerp == 0f)
        {
            middleRed.SetAnimationFrame(0);
            middleGreen.SetAnimationFrame(0);
        }

        if (target == 1)
        {
            speed = Calc.Approach(speed, maxForwardSpeed, maxForwardSpeed / 0.2f * Engine.DeltaTime);
        }
        else
        {
            speed = Calc.Approach(speed, maxBackwardSpeed, maxBackwardSpeed / 1.5f * Engine.DeltaTime);
        }

        float num = lerp;
        lerp = Calc.Approach(lerp, target, speed * Engine.DeltaTime);
        if (lerp != num)
        {
            Vector2 liftSpeed = (end - start) * speed;
            Vector2 position = Position;
            if (target == 1)
            {
                liftSpeed = (end - start) * maxForwardSpeed;
            }

            if (lerp < num)
            {
                liftSpeed *= -1f;
            }

            if (target == 1 && base.Scene.OnInterval(0.02f))
            {
                MoveParticles(end - start);
            }

            MoveTo(Vector2.Lerp(start, end, lerp), liftSpeed);
            if (position != Position)
            {
                Audio.Position(moveSfx, base.Center);
                Audio.Position(returnSfx, base.Center);
                if (Position == start && target == 0)
                {
                    Audio.SetParameter(returnSfx, "end", 1f);
                    Audio.Play("event:/game/05_mirror_temple/swapblock_return_end", base.Center);
                    Destroy();
                }
                else if (Position == end && target == 1)
                {
                    Audio.Play("event:/game/05_mirror_temple/swapblock_move_end", base.Center);
                }
            }
        }

        if (Swapping && lerp >= 1f)
        {
            Swapping = false;
        }

        StopPlayerRunIntoAnimation = lerp <= 0f || lerp >= 1f;
    }
    public void Destroy()
    {
        path.RemoveSelf();
        Break(Position);
        RemoveSelf();
    }

    public void Break(Vector2 from)
    {
        Audio.Play("event:/game/general/wall_break_stone", Position);

        for (int i = 0; (float)i < base.Width / 8f; i++)
        {
            for (int j = 0; (float)j < base.Height / 8f; j++)
            {
                base.Scene.Add(Engine.Pooler.Create<Debris>().Init(Position + new Vector2(4 + i * 8, 4 + j * 8), '1', true).BlastFrom(from));
            }
        }
    }
    private void MoveParticles(Vector2 normal)
    {
        Vector2 position;
        Vector2 positionRange;
        float direction;
        float num;
        if (normal.X > 0f)
        {
            position = base.CenterLeft;
            positionRange = Vector2.UnitY * (base.Height - 6f);
            direction = MathF.PI;
            num = Math.Max(2f, base.Height / 14f);
        }
        else if (normal.X < 0f)
        {
            position = base.CenterRight;
            positionRange = Vector2.UnitY * (base.Height - 6f);
            direction = 0f;
            num = Math.Max(2f, base.Height / 14f);
        }
        else if (normal.Y > 0f)
        {
            position = base.TopCenter;
            positionRange = Vector2.UnitX * (base.Width - 6f);
            direction = -MathF.PI / 2f;
            num = Math.Max(2f, base.Width / 14f);
        }
        else
        {
            position = base.BottomCenter;
            positionRange = Vector2.UnitX * (base.Width - 6f);
            direction = MathF.PI / 2f;
            num = Math.Max(2f, base.Width / 14f);
        }

        particlesRemainder += num;
        int num2 = (int)particlesRemainder;
        particlesRemainder -= num2;
        positionRange *= 0.5f;
        SceneAs<Level>().Particles.Emit(SwapBlock.P_Move, num2, position, positionRange, direction);
    }

    [MethodImpl(MethodImplOptions.NoInlining)]
    public override void Render()
    {
        Vector2 vector = Position + base.Shake;
        if (lerp != (float)target && speed > 0f)
        {
            Vector2 vector2 = (end - start).SafeNormalize();
            if (target == 1)
            {
                vector2 *= -1f;
            }

            float num = speed / maxForwardSpeed;
            float num2 = 16f * num;
            for (int i = 2; (float)i < num2; i += 2)
            {
                DrawBlockStyle(vector + vector2 * i, base.Width, base.Height, nineSliceGreen, middleGreen, Color.White * (1f - (float)i / num2));
            }
        }

        if (redAlpha < 1f)
        {
            DrawBlockStyle(vector, base.Width, base.Height, nineSliceGreen, middleGreen, Color.White);
        }

        if (redAlpha > 0f)
        {
            DrawBlockStyle(vector, base.Width, base.Height, nineSliceRed, middleRed, Color.White * redAlpha);
        }
    }

    [MethodImpl(MethodImplOptions.NoInlining)]
    private void DrawBlockStyle(Vector2 pos, float width, float height, MTexture[,] ninSlice, Sprite middle, Color color)
    {
        int num = (int)(width / 8f);
        int num2 = (int)(height / 8f);
        ninSlice[0, 0].Draw(pos + new Vector2(0f, 0f), Vector2.Zero, color);
        ninSlice[2, 0].Draw(pos + new Vector2(width - 8f, 0f), Vector2.Zero, color);
        ninSlice[0, 2].Draw(pos + new Vector2(0f, height - 8f), Vector2.Zero, color);
        ninSlice[2, 2].Draw(pos + new Vector2(width - 8f, height - 8f), Vector2.Zero, color);
        for (int i = 1; i < num - 1; i++)
        {
            ninSlice[1, 0].Draw(pos + new Vector2(i * 8, 0f), Vector2.Zero, color);
            ninSlice[1, 2].Draw(pos + new Vector2(i * 8, height - 8f), Vector2.Zero, color);
        }

        for (int j = 1; j < num2 - 1; j++)
        {
            ninSlice[0, 1].Draw(pos + new Vector2(0f, j * 8), Vector2.Zero, color);
            ninSlice[2, 1].Draw(pos + new Vector2(width - 8f, j * 8), Vector2.Zero, color);
        }

        for (int k = 1; k < num - 1; k++)
        {
            for (int l = 1; l < num2 - 1; l++)
            {
                ninSlice[1, 1].Draw(pos + new Vector2(k, l) * 8f, Vector2.Zero, color);
            }
        }

        if (middle != null)
        {
            middle.Color = color;
            middle.RenderPosition = pos + new Vector2(width / 2f, height / 2f);
            middle.Render();
        }
    }
}