local utils = require("utils")
local roomStruct = require("structs.room")
local state = require("loaded_state")  -- To access the current map state

local script = {
    name = "copyAllRooms",
    displayName = "Copy All Rooms",
    tooltip = "Copy all rooms in the current map to your clipboard"
}

function script.run(_, args)
    if not state.map or not state.map.rooms then
        print("No map or rooms found")
        return  -- Exit if no map or rooms found
    end

    local allRooms = state.map.rooms
    local serializedRooms = {}

    for _, room in ipairs(allRooms) do
        -- This assumes roomStruct.encode includes everything (structure + content)
        -- If this is not the case, we need to manually serialize the contents as well
        local encodedRoom = roomStruct.encode(room)  
        local success, text = utils.serialize(encodedRoom, nil, nil, false)  -- Serialize entire room (structure + contents)
        if success then
            table.insert(serializedRooms, text)
        else
            print("Failed to serialize room:", room, "with data:", encodedRoom)  -- Log if serialization fails
        end
    end

    -- Combine all serialized rooms into a single string, separated by "+"
    local combinedText = table.concat(serializedRooms, "+")

    -- Debug: Check the contents before copying to clipboard
    print("Serialized Rooms:")
    for i, roomText in ipairs(serializedRooms) do
        print(i, roomText)  -- Log each serialized room's data
    end

    -- Copy the combined serialized room data to the clipboard
    love.system.setClipboardText(combinedText)
    print("Serialized data copied to clipboard")
end

return script