local utils = require("utils")
local map_item_utils = require("map_item_utils")
local loadedState = require("loaded_state")
local roomStruct = require("structs.room")

local script = {
    name = "pasteAllRooms",
    displayName = "Paste All Rooms",
    tooltip = "Paste rooms with an immediate offset if they already exist",
    parameters = {
        nameOffset = "",
        offsetAmount = 5000,
        amount = 1
    }
}

function script.form()
    return {
        nameOffset = {
            label = "Name Suffix",
            type = "string"
        },
        offsetAmount = {
            label = "Offset Distance (px)",
            type = "integer"
        },
        amount = {
            label = "Max Duplicates to Add",
            type = "integer"
        }
    }
end

local function split(inputstr, sep)
    if sep == nil then sep = "%s" end
    local t = {}
    for str in string.gmatch(inputstr, "([^"..sep.."]+)") do
        table.insert(t, str)
    end
    return t
end

function script.run(_, args)
    local map = loadedState.map
    if not map then return end

    local nameset = args.nameOffset
    local offset = args.offsetAmount
    local maxVersions = args.amount

    local clipboardText = love.system.getClipboardText()
    if not clipboardText or clipboardText == "" then return end

    local roomDataList = split(clipboardText, "+")

    for _, roomText in ipairs(roomDataList) do
        if roomText == "" then goto continue end

        local success, roomData = utils.unserialize(roomText)
        if success and type(roomData) == "table" then
            local newRoom = roomStruct.decode(roomData)

            if newRoom then
                local duplicateCount = 0
                local baseName = newRoom.name
                local currentName = baseName
                
                -- Check if the room exists. If it does, we start incrementing.
                while loadedState.getRoomByName(currentName) do
                    duplicateCount = duplicateCount + 1
                    
                    -- Construct the new name based on suffix and count
                    if nameset ~= "" then
                        currentName = string.format("%s (%d) [%s]", baseName, duplicateCount, nameset)
                    else
                        currentName = string.format("%s (%d)", baseName, duplicateCount)
                    end
                end

                -- If duplicateCount > 0, it means the room existed, so we apply the offset
                if duplicateCount > 0 and duplicateCount <= maxVersions then
                    newRoom.name = currentName
                    
                    -- Apply offset: 1st duplicate gets 1*5000, 2nd gets 2*5000, etc.
                    newRoom.x = newRoom.x + (offset * duplicateCount)
                    newRoom.y = newRoom.y + (offset * duplicateCount)
                    
                    map_item_utils.addRoom(map, newRoom)
                    print("Room duplicated at offset: " .. currentName)
                elseif duplicateCount == 0 then
                    -- If the room didn't exist at all, just add it normally
                    map_item_utils.addRoom(map, newRoom)
                    print("Room added (no offset needed): " .. newRoom.name)
                end
            end
        end
        ::continue::
    end

    if loadedState.refreshMap then
        loadedState.refreshMap()
    end
end

return script